package ru.scancode.GodexPrintTest.utils;

import android.graphics.Bitmap;
import android.graphics.Color;
import android.util.Log;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;

public class MonochromeBMPUtil {

    // кратность (каждая строка должна быть кратна 4м байтам)
    private static final int BMP_WIDTH_OF_TIMES = 4;
    // пороговое значение для монохрома
    private static final int DEFAULT_MONOCHROME_GATE = 80;


    public static boolean save(Bitmap orgBitmap, String filePath) {
        return save(orgBitmap, filePath, DEFAULT_MONOCHROME_GATE);
    }

    public static boolean save(Bitmap orgBitmap, String filePath, int inGate) {
        boolean isSaveSuccess = false;
        long start = System.currentTimeMillis();
        inGate = (inGate > 255 || inGate < 0) ? DEFAULT_MONOCHROME_GATE : inGate;
        if (orgBitmap == null) {
            return isSaveSuccess;
        }

        if (filePath == null) {
            return isSaveSuccess;
        }

        //размер изображения
        int width = orgBitmap.getWidth();
        int height = orgBitmap.getHeight();

        // рассчитаем добавку к строке, тк количество байт должно быть кратно 4
        byte[] dummyBytesPerRow = null;
        boolean hasDummyBytes = false;
        boolean hasDummyBits = false;

        // количество байт на строку
        int rowWidthInBytes;
        int dummyBits;
        if (width % 8 == 0) {
            dummyBits = 0;
            rowWidthInBytes = width / 8;
        } else {
            hasDummyBits = true;
            rowWidthInBytes = width / 8 + 1;
            dummyBits = rowWidthInBytes * 8 - width;
        }

        if (rowWidthInBytes % BMP_WIDTH_OF_TIMES > 0) {
            hasDummyBytes = true;
            //количество байт для добавки
            dummyBytesPerRow = new byte[(BMP_WIDTH_OF_TIMES - (rowWidthInBytes % BMP_WIDTH_OF_TIMES))];
            //заполнение неиспользуемых байтов в конец каждой строки
            for (int i = 0; i < dummyBytesPerRow.length; i++) {
                dummyBytesPerRow[i] = (byte) 0x00;
            }
        }

        //массив исходных пикселей
        int[] pixels = new int[width * height];

        // количество байт, необходимое для хранения информации не включая заголовка
        int imageSize = (rowWidthInBytes + (hasDummyBytes ? dummyBytesPerRow.length : 0)) * height;
        //размер заголовка
        int imageDataOffset = 0x3E;

        //общий размер
        int fileSize = imageSize + imageDataOffset;

        //исходный Android Bitmap Image Data (- получение массива пикселей из него)
        orgBitmap.getPixels(pixels, 0, width, 0, 0, width, height);

        //создаем буфер байт для записи сконвертированного изображения
        ByteBuffer buffer = ByteBuffer.allocate(fileSize);

        /*
          BITMAP FILE HEADER Write Start
         */
        buffer.put((byte) 0x42);
        buffer.put((byte) 0x4D);

        //size
        buffer.put(writeInt(fileSize));

        //reserved
        buffer.put(writeShort((short) 0));
        buffer.put(writeShort((short) 0));

        //image data start offset
        buffer.put(writeInt(imageDataOffset));

        /* BITMAP FILE HEADER Write End */

        //*************************************

        /* BITMAP INFO HEADER Write Start */
        //size
        buffer.put(writeInt(0x28));

        //width, height
        buffer.put(writeInt(width));
        buffer.put(writeInt(height));

        //planes
        buffer.put(writeShort((short) 1));

        //bit count - разрешение - бит на пиксель
        buffer.put(writeShort((short) 1));

        //bit compression
        buffer.put(writeInt(0));

        //image data size
        buffer.put(writeInt(imageSize));

        //horizontal resolution in pixels per meter
        buffer.put(writeInt(0));

        //vertical resolution in pixels per meter (unreliable)
        buffer.put(writeInt(0));
        buffer.put(writeInt(0));
        buffer.put(writeInt(0));

        // test add
        buffer.put(writeInt(0));
        buffer.put((byte) 0xFF);
        buffer.put((byte) 0xFF);
        buffer.put((byte) 0xFF);
        buffer.put((byte) 0x00);

        /* BITMAP INFO HEADER Write End */

        // ********************************

        /* BITMAP IMAGE Write Start */

        int row = height;
        int col = width;
        int startPosition = (row - 1) * col;
        int endPosition = row * col;
        boolean fullByte = false;
        int bitCounter = 0;
        byte workByte;
        workByte = (byte) 0;

        while (row > 0) {
            for (int i = startPosition; i < endPosition; i++) {

                // например для монохромного рисунка (из входящего массива пикселей) :
                // (-16777216) - черный / (-1) - белый

                // здесь 1 - полезная точка (черный) - т.е. делаем ++
                if (getMonoColor(pixels[i], inGate) == 0) workByte++;


                bitCounter++;
                if (bitCounter == 8) fullByte = true;
                if (fullByte) {
                    Log.i("TAG", "workByte = " + workByte);
                    buffer.put(workByte);
                    workByte = 0;
                    fullByte = false;
                    bitCounter = 0;
                } else {
                    workByte = (byte) (workByte << 1);
                }
            }
            // добиваем биты до полного байта
            if (hasDummyBits) {
                workByte = (byte) (workByte << (dummyBits - 1));
                Log.i("TAG", "workByte = " + workByte);
                Log.i("TAG", "added dummyBits = " + dummyBits);
                buffer.put(workByte);
                fullByte = false;
                bitCounter = 0;
                workByte = 0;
            }
            // добиваем байты до кратности четырем
            if (hasDummyBytes) {
                Log.i("TAG", "add dummyBytes = " + dummyBytesPerRow.length);
                buffer.put(dummyBytesPerRow);
            }
            row--;
            endPosition = startPosition;
            startPosition = startPosition - col;
            Log.i("TAG", "row = " + row);
        }

        /* BITMAP IMAGE Write End */

        // Запись файла
        FileOutputStream fos;
        try {
            fos = new FileOutputStream(filePath);
            fos.write(buffer.array());
            fos.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
        File file = new File(filePath);
        if (file.exists() && file.length() != 0L) {
            Log.i("TAG","size = " + file.length());
            isSaveSuccess = true;
        }
        Log.v("LOGTAG", System.currentTimeMillis() - start + " ms");

        return isSaveSuccess;
    }

    //Запись прямого порядка байт для int
    private static byte[] writeInt(int value) {
        byte[] b = new byte[4];
        b[0] = (byte) (value & 0x000000FF);
        b[1] = (byte) ((value & 0x0000FF00) >> 8);
        b[2] = (byte) ((value & 0x00FF0000) >> 16);
        b[3] = (byte) ((value & 0xFF000000) >> 24);
        return b;
    }

    //Запись прямого порядка байт для short
    private static byte[] writeShort(short value) {
        byte[] b = new byte[2];
        b[0] = (byte) (value & 0x00FF);
        b[1] = (byte) ((value & 0xFF00) >> 8);
        return b;
    }

    private static int getMonoColor(int inColor, int monochromeGate) {
        // 0 - белый, 1 - черный
        int outColor;

        int blue = Color.blue(inColor);
        int red = Color.red(inColor);
        int green = Color.green(inColor);
        int alpha = Color.alpha(inColor);

        // считаем цвет
        // все составляющие цвета больше гейта, а так же и если фон прозрачный (альфа не равен 255) - то белый
        if (blue > monochromeGate &&
                red > monochromeGate &&
                green > monochromeGate ||
                alpha != 255) {
            outColor = 0;
        } else {
            outColor = 1;
        }

        return outColor;
    }
}

